<?php
// Enable error reporting for debugging (can be disabled in production)
ini_set('display_errors', 1);
ini_set('display_startup_errors', 1);
error_reporting(E_ALL);

// Load required configuration and utility files
require_once 'functions.php';

// =============================================================================
// LOGOUT HANDLING
// =============================================================================

// Handle logout POST request (destroys session and redirects to homepage)
if (isset($_POST['logout'])) {
    logoutUser();
    header('Location: index.php');
    exit;
}

// =============================================================================
// LANGUAGE AND USER DATA SETUP
// =============================================================================

// Get current language (auto-detect from browser or use session)
$lang = get_default_lang();

// Load all users from walkers.json and filter out 'admin' user
// Admin user should not appear in the public walkers list
$users = array_filter(getUsers(), function($u) { 
    return $u['username'] !== 'admin'; 
});

// =============================================================================
// SORT USERS BY JOIN DATE (Newest First)
// =============================================================================

// Sort users by joined date in descending order (newest users appear first)
usort($users, function($a, $b) {
    $ta = isset($a['joined']) ? strtotime($a['joined']) : 0;  // Convert date to timestamp
    $tb = isset($b['joined']) ? strtotime($b['joined']) : 0;
    return $tb <=> $ta;  // Spaceship operator: newer dates first
});

// =============================================================================
// PAGINATION SETUP
// =============================================================================

// Get pagination settings and calculate page data
$users_per_page = getSetting('SITE_SETTINGS.principles_per_page', 10);  // Reuse principles setting
$page = isset($_GET['page']) ? max(1, intval($_GET['page'])) : 1;      // Current page (minimum 1)
$total_users = count($users);                                            // Total number of users
$total_pages = ceil($total_users / $users_per_page);                    // Calculate total pages
$offset = ($page - 1) * $users_per_page;                                // Starting index for current page

// Slice the users array to show only users for the current page
$users = array_slice($users, $offset, $users_per_page);
?>
<!DOCTYPE html>
<html lang="<?php echo $lang; ?>">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Walkers</title>
    <link rel="preconnect" href="https://fonts.googleapis.com">
    <link rel="preconnect" href="https://fonts.gstatic.com" crossorigin>
    <link href="https://fonts.googleapis.com/css2?family=Roboto:ital,wght@0,100..900;1,100..900&display=swap" rel="stylesheet">
    <link rel="stylesheet" href="style.css">
    <link rel="stylesheet" href="style.php">
    <style>
        body { background: #e3e6f5; }
        
        @media screen and (min-width: 768px) {
            .walkers-page{
                /* padding-right: 15px; */
            }
        }
        
        .walkers-container {
            max-width: 400px;
            margin: 20px auto 40px auto;
            background: #fff3d6;
            border-radius: 30px;
            box-shadow: <?php echo getSetting('PAGES.walkers.container_box_shadow', '0 4px 15px rgba(0,0,0,0.1)'); ?>;
            padding: 15px 15px 25px 15px;
        }
        .walkers-table {
            width: 100%;
            border-collapse: separate;
            border-spacing: 0 5px;
        }
        .walkers-table th, .walkers-table td {
            padding: 12.5px 10px;
            background: #fff;
            font-size: 0.75rem;
            color: #333;
            text-align: center;
        }
        .walkers-table th {
            background: #8fd3ff;
            color: #fff;
            font-size: 0.75rem;
        }
        .walkers-table th a {
            color: #fff;
            text-decoration: none;
            /* display: block; */
            /* transition: background-color 0.2s; */
        }
        .walkers-table thead tr th:first-child {
            border-top-left-radius: 12px;
            border-bottom-left-radius: 12px;
        }

        .walkers-table thead tr th:nth-child(2) {
            border-top-right-radius: 12px;
            border-bottom-right-radius: 12px;
        }
        
        .walkers-table tbody tr td:first-child {
            border-top-left-radius: 12px;
            border-bottom-left-radius: 12px;
        }

        .walkers-table tbody tr td:nth-child(2) {
            width: 55%;
            border-top-right-radius: 12px;
            border-bottom-right-radius: 12px;
      }
        .walker-username-link {
            color: #ffffff;
            text-decoration: underline;
            font-weight: bold;
            cursor: pointer;
        }
        .nav-container {
            width: 100%;
            max-width: 900px;
            margin: 0 auto;
        }
        .nav-left{
            display: flex;
            width: 100%;
            gap: 0;
        }
        .responsive-table-wrapper {
            overflow-x: auto;
        }

        /* .walkers-table tbody tr td {background: #7dcf95;color:#ffffff}
        .walkers-table tbody tr:nth-child(2n) td { background: #d29ec5; }
        .walkers-table tbody tr:nth-child(3n) td { background: #9E9E9E; }
        .walkers-table tbody tr:nth-child(4n) td { background: #fdac62; }
        .walkers-table tbody tr:nth-child(5n) td { background: #dcb35a; } */

        @media screen and (max-width: 900px) {
            .walkers-container{
                margin-top: 0;
            }
        }
        @media screen and (max-width: 768px) {
            .walkers-container{
                padding: 15px;
                margin: 0 5px 40px;
            }
            .walkers-table th,.walkers-table td{
                font-size: 0.75rem;
            }
        }
        /* Pagination styles (customizable) */
        .pagination {
            display: flex;
            justify-content: center;
            gap: 10px;
            margin-top: 15px;
        }
        .pagination .pagination-btn {
            background: <?php echo getSetting('PAGES.walkers.pagination.button_bg', getSetting('COLORS.navbar_button_bg_color', '#8FD3FF')); ?>;
            color: <?php echo getSetting('PAGES.walkers.pagination.button_text', getSetting('COLORS.navbar_button_text_color', '#ffffff')); ?>;
        }
    </style>
</head>
<body class="walkers-page">
<nav class="navbar">
    <div class="nav-container nav-flex">
        <div class="nav-left">
            <a href="index.php" class="nav-button"><?php echo t('prescription', $lang); ?></a>
            <a href="principles.php" class="nav-button"><?php echo t('principles', $lang); ?></a>
            <a href="walkers.php" class="nav-button"><?php echo t('walkers', $lang); ?></a>
            <?php if (isLoggedIn()): ?>
                <a href="profile.php" class="nav-button profile-btn"><?php echo t('profile', $lang); ?></a>
            <?php else: ?>
                <a href="go.php" class="nav-button login-btn"><?php echo t('login_signup_topbar', $lang); ?></a>
            <?php endif; ?>
        </div>
    </div>
</nav>
<div class="walkers-container">
    <div class="responsive-table-wrapper">
        <table class="walkers-table">
            
            <tbody>
            <?php if (empty($users)): ?>
                <tr>
                    <span style="text-align:center; padding: 40px; font-size: 1.2rem; color: #888;">
                        No walkers have joined yet. Be the first!
                    </span>
                </tr>
            <?php else: ?>
                <?php foreach ($users as $user): ?>
                    <tr>
                        <td><a href="profile?user=<?php echo urlencode($user['username']); ?>" class="walker-username-link"><?php echo htmlspecialchars($user['username']); ?></a></td>
                        <td><?php echo date('m/y', strtotime($user['joined'])); ?></td>
                    </tr>
                <?php endforeach; ?>
            <?php endif; ?>
            </tbody>
        </table>
    </div>
    
    <?php if ($total_pages > 1): ?>
    <?php
        $disabled_bg = getSetting('PAGES.walkers.pagination.button_bg_disabled', '#ccc');
        $disabled_text = getSetting('PAGES.walkers.pagination.button_text_disabled', '#999');
    ?>
    <div class="pagination">
        <a href="?page=<?php echo ($page - 1); ?>&lang=<?php echo $lang; ?>" class="pagination-btn" <?php if ($page <= 1) echo 'style="pointer-events:none;background:'.$disabled_bg.';color:'.$disabled_text.';"'; ?>>&#8592;</a>
        <a href="?page=<?php echo ($page + 1); ?>&lang=<?php echo $lang; ?>" class="pagination-btn" <?php if ($page >= $total_pages) echo 'style="pointer-events:none;background:'.$disabled_bg.';color:'.$disabled_text.';"'; ?>>&#8594;</a>
    </div>
    <?php endif; ?>
</div>

</body>
</html> 