<?php
/**
 * PRINCIPLES WEBSITE - MAIN CONFIGURATION FILE
 * ===============================================
 * 
 * This is the SINGLE SOURCE OF TRUTH for all website configuration!
 * Everything is centralized here for maximum customization and control.
 * 
 * WHAT YOU CAN CUSTOMIZE:
 * - Site name and basic information
 * - Moderator user IDs (who can approve/reject principles)
 * - Advanced color system: per-page text colors, button colors, backgrounds
 * - Complete bilingual interface text (English + one other language)
 * - Button labels, icons, and interface elements
 * - Timezone and regional settings
 * - Language detection and auto-switching behavior
 * - Font families and sizes for all text elements
 * 
 * HOW TO USE:
 * 1. Edit the values in the $SETTINGS array below
 * 2. Save this file
 * 3. Refresh your website - changes apply immediately
 * 4. No database required - all settings are file-based
 * 
 * IMPORTANT NOTES:
 * - Use single quotes for text values to avoid conflicts
 * - Test your changes on a staging site first if possible
 * - This file contains ALL configuration - no other config files needed
 * 
 * STRICT BILINGUAL SYSTEM (ENGLISH + ONE OTHER LANGUAGE ONLY):
 * - This system is designed to support EXACTLY TWO LANGUAGES: English + One Other
 * - English ('en') is always the primary language
 * - One configurable 'other' language (e.g., French, Spanish, German, etc.)
 * - Browser auto-detection chooses between these two languages automatically
 * - DO NOT add more than two languages - the system architecture is bilingual-only
 * - To change the 'other' language, modify 'other_language_code' below
 * - Landing page content is managed separately in parser.php
 */

// =============================================================================
// LANGUAGE SETTINGS - CONFIGURE THIS FIRST!
// =============================================================================
$SETTINGS = [
    'LANGUAGES' => [
        'auto_detect' => true,                 // Auto-detect from browser (true/false)
        'other_language_code' => 'fr',         // CHANGE THIS: Your language code (fr, es, de, it, pt, etc.)
        // NOTE: System always has English ('en') + one other language
        // Browser detection only shows "other" language if browser matches your code EXACTLY
        // Example: If set to 'fr' and user has German browser → shows English
        //          If set to 'fr' and user has French browser → shows French
    ],

    // =============================================================================
    // BASIC SITE INFORMATION
    // =============================================================================
    'SITE_SETTINGS' => [
        'site_name_eng' => 'Principles',                 // CHANGE THIS! English site name
        'site_name_other' => 'Principes',                // CHANGE THIS! Other language site name
        'timezone' => 'GMT+0',                           // Your timezone (GMT+5, GMT-3, etc.)
    ],

    // =============================================================================
    // MODERATORS - ADD YOUR USER_ID HERE!
    // =============================================================================
    'MODERATORS' => [
        '0000000000',   // CHANGE THIS to your user_id!
        // 'another_mod_id',   // Uncomment and add as needed
    ],

    // =============================================================================
    // DYNAMIC COLORS - CUSTOMIZE CARD AND ROW COLORS!
    // =============================================================================
    'COLORS' => [
        // Container background color (global default body only)
        'body_background_color' => '#e3e6f5',
        
        // Global default text color (used as fallback only)
        'text_color_global' => '#ffffff',
        
        // Left three navigation buttons (Prescription, Principles, Walkers)
        'navbar_button_text_color' => '#ffffff',  // Text color for left three nav buttons
        'navbar_button_bg_color' => '#8FD3FF',    // Background color for left three nav buttons
        'navbar_button_font_size' => '11px',      // Font size: 10px, 12px, 14px, 0.8rem, 1rem, etc.

        // Far right single button (Profile or Login/Go button)
        'profile_button_text_color' => '#ffffff', // Text color for far right button (Profile)
        'profile_button_bg_color' => '#f9a1b0',   // Background color for far right button (Profile)
        'profile_button_font_size' => '11px',     // Font size: 10px, 12px, 14px, 0.8rem, 1rem, etc.
        'login_button_text_color' => '#ffffff',   // Text color for far right button (Login/Go)
        'login_button_bg_color' => '#f9a1b0',     // Background color for far right button (Login/Go)
    ],

    'PAGES' => [
        // =============================================================================
        // PRESCRIPTION PAGE (Landing Page)
        // =============================================================================
        'prescription' => [
            'text_color' => '#000000',
            'container_background_color' => '#fff3d6',
            // Download button color
            'download_button_color' => '#8FD3FF',
            
            // Font customization
            // Font family examples:
            //   - Web-safe fonts: 'Arial, sans-serif', 'Georgia, serif', 'Courier New, monospace'
            //   - Google Fonts: 'Roboto, sans-serif', 'Open Sans, sans-serif', 'Lato, sans-serif'
            //   - System fonts: 'Inter, sans-serif', 'Segoe UI, sans-serif', '-apple-system, sans-serif'
            //   - Playful fonts: 'Comic Neue, cursive', 'Sniglet, cursive', 'Bubblegum Sans, cursive'
            'font_family' => 'Roboto, cursive',
            
            // Font size examples:
            //   - Pixels: '12px', '14px', '16px', '18px', '20px'
            //   - Rem units: '0.8rem', '1rem', '1.2rem', '1.5rem', '2rem'
            //   - Em units: '0.9em', '1em', '1.1em', '1.5em'
            'title_font_size' => '1.5rem',       // Title size (larger)
            'paragraph_font_size' => '0.7rem',   // Paragraph size (smaller)
        ],
        
        // =============================================================================
        // PRINCIPLES PAGE (Community Principles List)
        // =============================================================================
        'principles' => [
            'text_color' => '#ffffff',
            'container_background_color' => '#fff3d6',
            // Font family for principles page
            'font_family' => 'Roboto, sans-serif',
            // Pagination settings
            'principles_per_page' => 10,       // Number of principles to show per page
            'pagination' => [
                'button_bg' => '#8FD3FF',
                'button_text' => '#ffffff',
                'button_bg_disabled' => '#ccc',
                'button_text_disabled' => '#999',
            ],
            // Principles card colors (cycle through these 5 colors)
            'principles_cards' => [
                '#AFC1AC',
                '#CAA19C',
                '#9DA2C6',
                '#91B7BF',
                '#C5B8AE',
            ],
        ],
        
        // =============================================================================
        // WALKERS PAGE (User Directory)
        // =============================================================================
        'walkers' => [
            // Page text color
            'text_color' => '#ffffff',
            // Container background for walkers container
            'container_background_color' => '#fff3d6',
            // Walkers container box shadow
            'container_box_shadow' => '0 4px 15px rgba(0,0,0,0.1)',
            // Font family for walkers page
            'font_family' => 'Roboto, sans-serif',
            // Walkers page row colors (cycle through these 5 colors)
            'walkers_rows' => [
                '#AFC1AC',
                '#CAA19C',
                '#9DA2C6',
                '#91B7BF',
                '#C5B8AE',
            ],
            'pagination' => [
                'button_bg' => '#8FD3FF',
                'button_text' => '#ffffff',
                'button_bg_disabled' => '#ccc',
                'button_text_disabled' => '#999',
            ],
        ],
        
        // =============================================================================
        // LOGIN PAGE (Sign-Up / Login)
        // =============================================================================
        'signup' => [
            // Container background (auth box)
            'container_background_color' => '#ffffff',
            // Page text color
            'text_color' => '#333333',
            // Font family for signup page
            'font_family' => 'Roboto, sans-serif',
            // Authentication form colors
            'input_background_color' => '#f8f9fa',     // Username/password input background
            'input_border_color' => '#dee2e6',         // Input border color
            'input_text_color' => '#333333',           // Input text color
            'button_background_color' => '#f9a1b0',    // Signup/Login button background
            'button_text_color' => '#ffffff',          // Signup/Login button text
        ],
        
        // =============================================================================
        // PROFILE PAGE (User Profile & Principles Management)
        // =============================================================================
        'profile' => [
            'text_color' => '#000000',
            'container_background_color' => '#fff3d6',
            // Font family for profile page
            'font_family' => 'Roboto, sans-serif',
        ],
    ],

    // =============================================================================
    // PRINCIPLE SETTINGS - CONTENT MODERATION & LIMITS
    // =============================================================================
    'PRINCIPLE_SETTINGS' => [
        'title_word_limit' => 1,               // Maximum words allowed in principle titles (1-10)
        'description_char_limit' => 777,        // Maximum characters in principle descriptions
        'title_label_eng' => 'Principle',      // English label for title field
        'title_label_other' => 'Principe',     // Other language label for title field
        'description_label_eng' => 'Description',  // English label for description field
        'description_label_other' => 'Description', // Other language label for description field
        'normalize_spaces' => true,            // Whether to normalize multiple spaces to single space
    ],

    // =============================================================================
    // INTERFACE TEXT - ALL WEBSITE TEXT IN ENGLISH & "Other" Language
    // =============================================================================
    'INTERFACE_TEXT' => [
        'en' => [
            // Navigation bar menu 
            'prescription' => 'Prescription',
            'principles' => 'Principles',
            'walkers' => 'Walkers', 
            'profile' => 'Profile',
            'logout' => 'Logout',
            'login_signup_topbar' => 'Go',
            'login_signup_login_page' => 'Sign-Up / Login',
            
            // Authentication
            'login' => 'Login',
            'signup' => 'Sign Up',
            'username' => 'Username',
            'email' => 'Email',
            'password' => 'Password',
            
            // Alerts
            'password_mismatch' => '✗',
            
            // Error Messages
            'error_not_logged_in' => 'Not logged in.',
            'error_invalid_field' => 'Invalid field.',
            'error_user_not_found' => 'User not found.',
            'error_login_required' => 'Please log in to continue',
            'error_login_again' => 'User not found. Please log in again.',
            'error_edit_permission' => 'Please log in to edit profiles',
            'error_passwords_mismatch_short' => 'Passwords do not match or are too short.',
            'error_password_fields_required' => 'Please fill in all password fields.',
            'error_passwords_no_match' => 'New passwords do not match.',
            'error_password_length' => 'Password must be at least 6 characters long.',
            
            // Icons (using emoji for simplicity)
            'icon_edit' => '✏️',
            'icon_save' => '💾',
            'icon_cancel' => '❌',
            'icon_logout' => '⏻',
            
            // Profile Fields & Labels
            'joined' => 'Joined',
            
            // Messages & Alerts
            'login_required' => 'Please login to continue',
            
            // Moderation
            'no_principles' => 'No principles',
            
            // Units (for principle limits)
            'word' => 'word',
            'words' => 'words',
            'characters' => 'characters',
        ],
        
        'other' => [
            // Navigation
            'prescription' => 'Ordonnance',
            'principles' => 'Principes',
            'walkers' => 'Marcheurs',
            'profile' => 'Profil',
            'logout' => 'Déconnexion',
            'login_signup_topbar' => 'Connexion',
            'login_signup_login_page' => 'Inscription / Connexion',
            
            // Authentication
            'login' => 'Connexion',
            'signup' => 'Inscription',
            'username' => 'Nom d\'utilisateur',
            'email' => 'Email',
            'password' => 'Mot de passe',
            
            // Alerts
            'password_mismatch' => '✗',
            
            // Error Messages
            'error_not_logged_in' => 'Non connecté.',
            'error_invalid_field' => 'Champ invalide.',
            'error_user_not_found' => 'Utilisateur non trouvé.',
            'error_login_required' => 'Veuillez vous connecter pour continuer',
            'error_login_again' => 'Utilisateur non trouvé. Veuillez vous reconnecter.',
            'error_edit_permission' => 'Veuillez vous connecter pour modifier les profils',
            'error_passwords_mismatch_short' => 'Les mots de passe ne correspondent pas ou sont trop courts.',
            'error_password_fields_required' => 'Veuillez remplir tous les champs de mot de passe.',
            'error_passwords_no_match' => 'Les nouveaux mots de passe ne correspondent pas.',
            'error_password_length' => 'Le mot de passe doit contenir au moins 6 caractères.',
            
            // Icons (using emoji for simplicity)
            'icon_edit' => '✏️',
            'icon_save' => '💾',
            'icon_cancel' => '❌',
            'icon_logout' => '⏻',
            
            // Profile Fields & Labels
            'joined' => 'Rejoint',
            
            // Messages & Alerts
            'login_required' => 'Veuillez vous connecter pour continuer',
            
            // Moderation
            'no_principles' => 'Aucun principe',
            
            // Units (for principle limits)
            'word' => 'mot',
            'words' => 'mots',
            'characters' => 'caractères',
        ],
    ],
];

// =============================================================================
// HELPER FUNCTIONS - DON'T MODIFY BELOW THIS LINE
// =============================================================================

if (!function_exists('getSetting')) {
    /**
     * Get a setting value from the settings array using dot notation
     * 
     * Allows you to access nested settings using a simple string path.
     * Example: getSetting('PAGES.profile.text_color', '#000000')
     * 
     * Process:
     * 1. Takes a dot-separated key path (e.g., 'COLORS.navbar_button_bg_color')
     * 2. Splits the key into individual parts
     * 3. Traverses the settings array step by step
     * 4. Returns the found value or the default if not found
     * 
     * @param string $key Dot-separated path to the setting (e.g., 'PAGES.profile.text_color')
     * @param mixed $default Default value to return if setting is not found
     * @return mixed The setting value or the default value
     * 
     * Usage examples:
     * - getSetting('SITE_SETTINGS.site_name_eng') returns 'Principles'
     * - getSetting('COLORS.navbar_button_bg_color', '#8FD3FF') returns '#8FD3FF'
     * - getSetting('PAGES.profile.font_family') returns 'Roboto, sans-serif'
     */
    function getSetting($key, $default = null) {
        global $SETTINGS;
        $keys = explode('.', $key);  // Split 'PAGES.profile.text_color' into ['PAGES', 'profile', 'text_color']
        $value = $SETTINGS;
        
        // Navigate through each level of the array
        foreach ($keys as $k) {
            if (isset($value[$k])) {
                $value = $value[$k];  // Go one level deeper
            } else {
                return $default;  // Key not found, return default
            }
        }
        return $value;  // Found the value
    }
}

if (!function_exists('getAppText')) {
    /**
     * Get translated interface text for the current or specified language
     * 
     * Retrieves text strings from INTERFACE_TEXT based on the current language.
     * Automatically falls back to English if translation is not found.
     * 
     * Process:
     * 1. Determines the language (provided or auto-detected)
     * 2. Looks up the text key in the specified language
     * 3. If not found, falls back to English version
     * 4. If still not found, returns the key itself
     * 
     * @param string $key The text key to look up (e.g., 'login', 'username', 'error_login_required')
     * @param string|null $lang Optional language code ('en' or 'other'). If null, uses current language.
     * @return string The translated text string
     * 
     * Usage examples:
     * - getAppText('login') returns 'Login' (English) or 'Connexion' (French)
     * - getAppText('username', 'en') returns 'Username'
     * - getAppText('error_login_required') returns translated error message
     */
    function getAppText($key, $lang = null) {
        if ($lang === null) {
            $lang = getCurrentLanguage();  // Auto-detect current language
        }
        
        // Try to get text in requested language
        $text = getSetting("INTERFACE_TEXT.{$lang}.{$key}");
        
        // Fallback to English if not found
        if ($text === null) {
            $text = getSetting("INTERFACE_TEXT.en.{$key}", $key);
        }
        
        return $text;
    }
}

if (!function_exists('getCurrentLanguage')) {
    /**
     * Determine the current language for the user
     * 
     * Implements bilingual system logic (English + one other language only).
     * Uses session storage and browser auto-detection.
     * 
     * Process:
     * 1. Check if session already has a language set (user preference)
     * 2. If auto-detect is enabled, check browser's Accept-Language header
     * 3. Only show "other" language if browser matches the configured language code EXACTLY
     * 4. Otherwise, always default to English
     * 
     * Language Logic:
     * - System supports EXACTLY two languages: English ('en') + one 'other'
     * - 'other' language code is configured in LANGUAGES.other_language_code
     * - Example: If other_language_code = 'fr' and browser is French, returns 'other'
     * - Example: If other_language_code = 'fr' and browser is German, returns 'en' (fallback)
     * 
     * @return string Language code: 'en' for English, 'other' for the configured second language
     * 
     * Usage:
     * - getCurrentLanguage() returns 'en' or 'other'
     * - Used by all pages to determine which language to display
     */
    function getCurrentLanguage() {
        // Start session if not already started (needed for language storage)
        if (session_status() === PHP_SESSION_NONE) {
            session_start();
        }
        
        // Check if user has a language preference in their session
        if (isset($_SESSION['language'])) {
            return $_SESSION['language'];
        }
        
        // Auto-detect language from browser if enabled
        if (getSetting('LANGUAGES.auto_detect', true)) {
            // Get browser's Accept-Language header (e.g., 'fr-FR,fr;q=0.9,en-US;q=0.8')
            $browserLang = isset($_SERVER['HTTP_ACCEPT_LANGUAGE']) ? $_SERVER['HTTP_ACCEPT_LANGUAGE'] : '';
            $otherLangCode = getSetting('LANGUAGES.other_language_code');  // e.g., 'fr', 'es', 'de'
            
            // Check if browser language contains the configured "other" language code
            // Only show "other" language if there's an EXACT match
            if ($otherLangCode && strpos(strtolower($browserLang), strtolower($otherLangCode)) !== false) {
                return 'other';  // Browser matches configured language
            }
        }
        
        // Default to English for all other cases
        return 'en';
    }
}

if (!function_exists('isModeratorBySettings')) {
    /**
     * Check if a user is a moderator based on their user_id
     * 
     * Compares the provided user_id against the list of moderator IDs
     * defined in the MODERATORS setting.
     * 
     * Process:
     * 1. Gets the list of moderator user IDs from settings
     * 2. Checks if the provided user_id exists in that list
     * 3. Returns true if user is a moderator, false otherwise
     * 
     * Moderator Privileges:
     * - Can approve or reject pending principles
     * - Can see all pending principles on the principles page
     * - Regular users cannot see moderation controls
     * 
     * @param string $user_id The user's unique ID (e.g., '2970089823')
     * @return bool True if user is a moderator, false otherwise
     * 
     * Usage:
     * - if (isModeratorBySettings($user['user_id'])) { show moderation buttons }
     * - Used in principles.php to show approve/deny buttons
     */
    function isModeratorBySettings($user_id) {
        $moderators = getSetting('MODERATORS', []);  // Get array of moderator IDs from settings
        return in_array($user_id, $moderators);      // Check if user_id is in the list
    }
}

if (!function_exists('getSiteUrl')) {
    /**
     * Get the full base URL of the website
     * 
     * Constructs the complete URL including protocol (http/https), domain name,
     * and directory path where the site is installed.
     * 
     * Process:
     * 1. Detects if site is using HTTPS or HTTP
     * 2. Gets the server name (domain)
     * 3. Gets the directory path
     * 4. Combines them into a full URL
     * 
     * @return string The full base URL
     * 
     * Examples:
     * - http://example.com/principles
     * - https://mysite.org
     * - http://localhost/myproject
     * 
     * Usage:
     * - getSiteUrl() returns 'https://example.com/principles'
     * - Used for generating absolute URLs for links, redirects, or API calls
     */
    function getSiteUrl() {
        return sprintf(
            "%s://%s%s",
            isset($_SERVER['HTTPS']) && $_SERVER['HTTPS'] != 'off' ? 'https' : 'http',  // Protocol
            $_SERVER['SERVER_NAME'],        // Domain name
            dirname($_SERVER['REQUEST_URI'])  // Directory path
        );
    }
}

if (!function_exists('getSiteName')) {
    /**
     * Get the site name in the appropriate language
     * 
     * Returns the bilingual site name based on the current or specified language.
     * Supports both English and the configured "other" language.
     * 
     * Process:
     * 1. Determines the language (provided or auto-detected)
     * 2. Returns site_name_eng for English
     * 3. Returns site_name_other for the other configured language
     * 4. Falls back to 'Principles' if not configured
     * 
     * @param string|null $lang Optional language code ('en' or 'other'). If null, uses current language.
     * @return string The site name in the appropriate language
     * 
     * Examples:
     * - getSiteName('en') returns 'Principles'
     * - getSiteName('other') returns 'Principes' (if French is configured)
     * - getSiteName() auto-detects based on current language
     * 
     * Usage:
     * - echo getSiteName() displays site name in user's language
     * - Used in page titles, headers, and navigation
     */
    function getSiteName($lang = null) {
        if ($lang === null) {
            $lang = getCurrentLanguage();  // Auto-detect current language
        }
        
        // Return appropriate site name based on language
        if ($lang === 'other') {
            return getSetting('SITE_SETTINGS.site_name_other', 'Principles');  // Other language version
        } else {
            return getSetting('SITE_SETTINGS.site_name_eng', 'Principles');    // English version
        }
    }
}

if (!function_exists('setTimezoneFromGMT')) {
    /**
     * Convert simplified GMT offset to PHP timezone format
     * 
     * Converts user-friendly GMT offset notation (GMT+5, GMT-3) into the format
     * required by PHP's date/time functions.
     * 
     * Process:
     * 1. Parses the GMT offset string using regex (e.g., 'GMT+5', 'GMT-3')
     * 2. Extracts the numeric offset
     * 3. Converts to PHP timezone format (note: PHP uses opposite sign convention)
     * 4. Returns the appropriate PHP timezone string
     * 
     * Important: PHP uses OPPOSITE signs in Etc/GMT timezones
     * - GMT+5 (ahead of UTC) becomes Etc/GMT-5 in PHP
     * - GMT-3 (behind UTC) becomes Etc/GMT+3 in PHP
     * - GMT+0 becomes UTC
     * 
     * @param string $gmt_offset GMT offset in format 'GMT+5', 'GMT-3', or 'GMT+0'
     * @return string PHP timezone string compatible with date_default_timezone_set()
     * 
     * Examples:
     * - setTimezoneFromGMT('GMT+5') returns 'Etc/GMT-5'
     * - setTimezoneFromGMT('GMT-3') returns 'Etc/GMT+3'
     * - setTimezoneFromGMT('GMT+0') returns 'UTC'
     * - setTimezoneFromGMT('invalid') returns 'UTC' (fallback)
     * 
     * Usage:
     * - date_default_timezone_set(setTimezoneFromGMT('GMT+5'))
     */
    function setTimezoneFromGMT($gmt_offset) {
        // Parse GMT offset using regex (matches 'GMT+5', 'GMT-3', 'GMT0', etc.)
        if (preg_match('/^GMT([+-]?\d+)$/', $gmt_offset, $matches)) {
            $offset = intval($matches[1]);  // Extract numeric offset
            
            if ($offset == 0) {
                return 'UTC';  // GMT+0 is UTC
            } elseif ($offset > 0) {
                // Positive offset (ahead of UTC) uses NEGATIVE in PHP format
                return 'Etc/GMT-' . $offset;
            } else {
                // Negative offset (behind UTC) uses POSITIVE in PHP format
                return 'Etc/GMT+' . abs($offset);
            }
        }
        
        // Fallback if format is invalid
        return 'UTC';
    }
}
?>