<?php
// Enable error reporting for debugging (can be disabled in production)
ini_set('display_errors', 1);
ini_set('display_startup_errors', 1);
error_reporting(E_ALL);

// Load required configuration and utility files
require_once 'settings.php';
require_once 'functions.php';

// =============================================================================
// LOGOUT HANDLING
// =============================================================================

// Handle logout POST request (destroys session and redirects to homepage)
if (isset($_POST['logout'])) {
    logoutUser();
    header('Location: index.php');
    exit;
}

// =============================================================================
// LANGUAGE SETUP
// =============================================================================

// Get current language (auto-detect from browser or use session)
$lang = get_default_lang();

/**
 * Format datetime as month/year
 * 
 * Converts full datetime string to simple m/y format for display.
 * Example: "2024-01-15 10:30:00" becomes "01/24"
 * 
 * @param string $datetime Full datetime string
 * @return string Formatted date as "m/y"
 */
function timeAgo($datetime) {
    return date('m/y', strtotime($datetime));
}

// =============================================================================
// LOAD PRINCIPLES DATA
// =============================================================================

// Load all principles from JSON file
$principlesFile = __DIR__ . '/principles.json';
$all_principles_raw = file_exists($principlesFile) ? json_decode(file_get_contents($principlesFile), true) : [];
if (!is_array($all_principles_raw)) {
    $all_principles_raw = [];  // Ensure we always have an array
}

// =============================================================================
// DATA CLEANUP - Remove Orphaned Principles
// =============================================================================

// Automatically clean up principles whose authors (users) no longer exist.
// This prevents "Unknown" usernames from appearing on the page.
$cleaned_count = cleanupOrphanedPrinciples();
if ($cleaned_count > 0) {
    // Reload principles after cleanup to reflect changes
    $all_principles_raw = file_exists($principlesFile) ? json_decode(file_get_contents($principlesFile), true) : [];
    if (!is_array($all_principles_raw)) {
        $all_principles_raw = [];
    }
}

// =============================================================================
// MODERATION ACTIONS - Approve or Deny Principles
// =============================================================================

// Handle moderator actions (approve or deny) for pending principles.
// Only moderators can access this functionality (checked via isModerator).
if (isModerator() && $_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['principle_id'], $_POST['action'])) {
    $principle_id = $_POST['principle_id'];
    $action = $_POST['action'];  // Either 'approve' or 'deny'
    $found = false;

    // Search for the principle and update its status
    foreach ($all_principles_raw as &$p) {
        if ($p['id'] === $principle_id) {
            if ($action === 'approve') {
                $p['approved'] = true;  // Mark principle as approved
            }
            $found = true;
            break;
        }
    }

    // If action is 'deny', remove the principle entirely
    if ($found && $action === 'deny') {
        $all_principles_raw = array_filter($all_principles_raw, function ($p) use ($principle_id) {
            return $p['id'] !== $principle_id;  // Keep all except the denied one
        });
    }

    // Save changes to file and reload page
    if ($found) {
        file_put_contents($principlesFile, json_encode(array_values($all_principles_raw), JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE));
        header("Location: principles.php");  // Reload to show updated list
        exit;
    }
}

// =============================================================================
// FILTER AND ORGANIZE PRINCIPLES FOR DISPLAY
// =============================================================================

// Get pending principles (only for moderators)
// These are principles waiting for approval, shown at the top of the page.
$pending_principles = [];
if (isModerator()) {
    $pending_principles = array_filter($all_principles_raw, function ($p) {
        // Filter criteria: Not approved AND user still exists (prevent orphans)
        return empty($p['approved']) && isset($p['posted_by_user_id']) && getUserByUserId($p['posted_by_user_id']) !== null;
    });
    // Sort pending principles by creation date (newest first)
    usort($pending_principles, function($a, $b) {
        return strtotime($b['created_at']) <=> strtotime($a['created_at']);
    });
}

// Get approved principles (visible to everyone)
// These are the publicly visible principles that passed moderation.
$approved_principles = array_filter($all_principles_raw, function($p) {
    if (empty($p['approved'])) return false;  // Must be approved
    
    // Check if principle has user_id and user exists (data integrity)
    return isset($p['posted_by_user_id']) && getUserByUserId($p['posted_by_user_id']) !== null;
});
// Sort approved principles by creation date (newest first)
usort($approved_principles, function($a, $b) {
    return strtotime($b['created_at']) <=> strtotime($a['created_at']);
});

// Combine pending (top) and approved (bottom) for display
// Moderators see pending principles first, then approved ones
$principles_to_display = array_merge($pending_principles, $approved_principles);

// =============================================================================
// PAGINATION SETUP
// =============================================================================

// Calculate pagination based on total principles and per-page setting
$perPage = getSetting('PAGES.principles.principles_per_page', 10);  // Get from settings (default: 10)
$total = count($principles_to_display);                              // Total number of principles
$totalPages = max(1, ceil($total / $perPage));                       // Calculate total pages (minimum 1)
$page = isset($_GET['page']) && is_numeric($_GET['page']) ? max(1, min($totalPages, (int)$_GET['page'])) : 1;  // Current page
$start = ($page - 1) * $perPage;                                     // Starting index for current page
$principles_for_page = array_slice($principles_to_display, $start, $perPage);  // Get principles for current page only
?>
<!DOCTYPE html>
<html lang="<?php echo $lang; ?>">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title><?php echo t('principles'); ?></title>
    <link rel="preconnect" href="https://fonts.googleapis.com">
    <link rel="preconnect" href="https://fonts.gstatic.com" crossorigin>
    <link href="https://fonts.googleapis.com/css2?family=Roboto:ital,wght@0,100..900;1,100..900&display=swap" rel="stylesheet">
    <link rel="stylesheet" href="style.css" />
    <link rel="stylesheet" href="style.php" />
    <style>
        .principles-page .principles-grid .principle-card{
            margin-bottom: 10px;
        }
        .principles-grid {
            display: flex;
            flex-direction: column;
            gap: 15px;
            margin: 5px auto;
            max-width: 1200px;
        }
        .principle-card {
            border-radius: 20px;
            padding: 24px;
            box-shadow: 0 8px 20px rgba(0,0,0,0.08);
            background: #7dcf95;
            display: flex;
            align-items: flex-start;
            gap: 20px;
            transition: transform 0.2s;
            width: 100%;
            min-height: 315px;
            /* height: 260px; */
            overflow: hidden;
        }
        .principle-card.pending {
            min-height: 350px;
            /* height: 300px; */
        }
            
        .principle-card:nth-child(2n) { background: #d29ec5; }
        .principle-card:nth-child(3n) { background: #9E9E9E; }
        .principle-card:nth-child(4n) { background: #fdac62; }
        .principle-card:nth-child(5n) { background: #dcb35a; }
        
        .principle-avatar {
            width: 50px;
            height: 50px;
            border-radius: 50%;
            background: rgba(255,255,255,0.6);
            color: #fff;
            font-size: 1.8rem;
            display: flex;
            align-items: center;
            justify-content: center;
            font-weight: bold;
            flex-shrink: 0;
        }
        .principle-content {
            width: 100%;
            flex: 1;
            display: flex;
            flex-direction: column;
            height: 100%;
        }
        .principle-title {
            font-size: 1.15rem;
            color: #fff;
            margin: 0;
            font-weight: bold;
            overflow-wrap: anywhere;
        }
        .principle-desc {
            color: #fff;
            font-size: 0.75rem;
            margin-top: 12px;
            flex: 1;
            word-wrap: break-word;
        }
        .principle-meta {
            text-align: right;
            font-size: 0.65rem;
            color: rgba(255,255,255,0.8);
        }
        .principle-meta .meta-username {
            text-decoration: none;
            font-weight: bold;
            color: #fff;
        }
         .principle-meta .meta-username:hover {
            text-decoration: underline;
        }
        .principle-header {
            display: flex;
            justify-content: space-between;
            align-items: flex-start;
            width: 100%;
            margin-bottom: 8px;
        }
        /* Pagination styles (customizable) */
        .pagination {
            display: flex;
            justify-content: center;
            gap: 10px;
            margin-top: 15px;
        }
        .pagination .pagination-btn {
            background: <?php echo getSetting('PAGES.principles.pagination.button_bg', getSetting('COLORS.navbar_button_bg_color', '#8FD3FF')); ?>;
            color: <?php echo getSetting('PAGES.principles.pagination.button_text', getSetting('COLORS.navbar_button_text_color', '#ffffff')); ?>;
        }
        
        .pagination-info {
            color: #333;
            font-size: 1.05rem;
            margin: 0 8px;
        }
        .principle-card.pending {
            background: #ffebee; /* A light red to distinguish */
            border: 2px solid #e57373;
        }
        .principle-card.pending .principle-title,
        .principle-card.pending .principle-desc,
        .principle-card.pending .principle-meta,
        .principle-card.pending .meta-username {
            color: #333;
        }
        .moderation-actions {
            display: flex;
            gap: 15px;
            margin-top: auto;
            padding-top: 10px;
        }
        .moderation-btn {
            border: none;
            border-radius: 50%;
            width: 30px;
            height: 30px;
            display: flex;
            padding: 0;
            align-items: center;
            justify-content: center;
            font-size: 1.3rem;
            cursor: pointer;
            margin: 0 2px;
            transition: background 0.2s, color 0.2s;
        }
        .moderation-btn.approve {
            background: #2ecc40;
            color: #fff;
        }
        .moderation-btn.deny {
            background: #b71c1c;
            color: #fff;
        }
        .deny-btn-wrapper {
            background-color: #ffdddd;
            border-radius: 5px;
            padding: 0 4px;
            display: inline-block;
        }
        .nav-container {
            width: 100%;
            max-width: 900px;
            margin: 0 auto;
        }
        .nav-left {
            display: flex;
            width: 100%;
            gap: 0;
        }
    </style>
</head>
<body class="principles-page">
<nav class="navbar">
    <div class="nav-container nav-flex">
        <div class="nav-left">
            <a href="index.php" class="nav-button"><?php echo t('prescription', $lang); ?></a>
            <a href="principles.php" class="nav-button"><?php echo t('principles', $lang); ?></a>
            <a href="walkers.php" class="nav-button"><?php echo t('walkers', $lang); ?></a>
            <?php if (isLoggedIn()): ?>
                <a href="profile.php" class="nav-button profile-btn"><?php echo t('profile', $lang); ?></a>
            <?php else: ?>
                <a href="go.php" class="nav-button login-btn"><?php echo t('login_signup_topbar', $lang); ?></a>
            <?php endif; ?>
        </div>
    </div>
</nav>

<main class="container">
    <div class="principles-bg">
        <div class="principles-grid">

            <?php if (empty($principles_to_display)): ?>
                <div class="no-principles-message" style="text-align: center; padding: 50px; font-size: 1.3rem; color: #777; border-radius: 20px;">
                    <?php if (isLoggedIn()): ?>
                        None
                    <?php else: ?>
                        <br><a href="go.php" style="color: #4a90e2; text-decoration: underline; margin-top: 10px; display: inline-block;">Log in to share a principle.</a>
                    <?php endif; ?>
                </div>
            <?php else: ?>
                <?php foreach ($principles_for_page as $p): ?>
                <?php 
                    $is_pending = !(isset($p['approved']) && $p['approved']);
                    $tp = translate_principle($p, $lang); 
                    
                    // Get user by user_id
                    $user = getUserByUserId($p['posted_by_user_id']);
                    
                    $card_class = $is_pending ? 'principle-card pending' : 'principle-card';
                ?>
                <div class="principle-card <?php echo $card_class; ?>">
                    <div class="principle-content">
                        <div class="principle-header">
                            <div class="principle-title">
                                <?php echo htmlspecialchars($tp['title']); ?>
                            </div>
                            <div class="principle-meta">
                                <?php 
                                $username = $user ? $user['username'] : ($tp['posted_by'] ?? 'Unknown');
                                if ($username && $username !== 'Unknown'): 
                                ?>
                                <a href="profile?user=<?php echo urlencode($username); ?>" class="meta-username">
                                    <?php echo htmlspecialchars($username); ?>
                                </a>
                                <?php else: ?>
                                <span class="meta-username">Unknown</span>
                                <?php endif; ?>
                                <span class="time-ago" title="<?php echo $tp['created_at']; ?>"> <?php echo timeAgo($tp['created_at']); ?></span>
                            </div>
                        </div>
                        <div class="principle-desc">
                            <?php echo nl2br(htmlspecialchars($tp['description'])); ?>
                        </div>
                        <?php if ($is_pending): ?>
                        <div class="moderation-actions">
                            <form method="post" action="principles" style="display:inline;">
                                <input type="hidden" name="principle_id" value="<?php echo htmlspecialchars($p['id']); ?>">
                                <button type="submit" name="action" value="approve" class="moderation-btn approve" title="Approve">
                                    <svg width="18" height="18" viewBox="0 0 20 20" fill="none" xmlns="http://www.w3.org/2000/svg">
                                        <path d="M5 10.5L9 14.5L15 7.5" stroke="white" stroke-width="2.2" stroke-linecap="round" stroke-linejoin="round"/>
                                    </svg>
                                </button>
                            </form>
                            <form method="post" action="principles" style="display:inline;">
                                <input type="hidden" name="principle_id" value="<?php echo htmlspecialchars($p['id']); ?>">
                                <button type="submit" name="action" value="deny" class="moderation-btn deny" title="Deny">
                                    <svg width="16" height="16" viewBox="0 0 20 20" fill="none" xmlns="http://www.w3.org/2000/svg">
                                        <path d="M6 6L14 14M14 6L6 14" stroke="white" stroke-width="2.2" stroke-linecap="round"/>
                                    </svg>
                                </button>
                            </form>
                        </div>
                        <?php endif; ?>
                    </div>
                </div>
                <?php endforeach; ?>
            <?php endif; ?>
        </div>
        <?php if (!empty($principles_to_display) && $totalPages > 1): ?>
        <?php
            $disabled_bg = getSetting('PAGES.principles.pagination.button_bg_disabled', '#ccc');
            $disabled_text = getSetting('PAGES.principles.pagination.button_text_disabled', '#999');
        ?>
        <div class="pagination">
            <a href="?page=<?php echo ($page - 1); ?>&lang=<?php echo $lang; ?>" class="pagination-btn" <?php if ($page <= 1) echo 'style="pointer-events:none;background:'.$disabled_bg.';color:'.$disabled_text.';"'; ?>>&#8592;</a>
            <a href="?page=<?php echo ($page + 1); ?>&lang=<?php echo $lang; ?>" class="pagination-btn" <?php if ($page >= $totalPages) echo 'style="pointer-events:none;background:'.$disabled_bg.';color:'.$disabled_text.';"'; ?>>&#8594;</a>
        </div>
        <?php endif; ?>
    </div>
</main>
</body>
</html>